﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using Tessa.BusinessCalendar;
using Tessa.Cards;
using Tessa.Cards.ComponentModel;
using Tessa.Extensions.Default.Server.Cards;
using Tessa.Extensions.Default.Server.Workflow;
using Tessa.Extensions.Default.Server.Workflow.WorkflowEngine;
using Tessa.Extensions.Default.Shared.Workflow.KrProcess;
using Tessa.Extensions.WorkflowExamples.Shared.Roles;
using Tessa.Extensions.WorkflowExamples.Shared.Workflow;
using Tessa.Localization;
using Tessa.Platform;
using Tessa.Platform.Data;
using Tessa.Platform.Storage;
using Tessa.Platform.Validation;
using Tessa.Workflow;
using Tessa.Workflow.Compilation;
using Tessa.Workflow.Helpful;
using Tessa.Workflow.Signals;

namespace Tessa.Extensions.WorkflowExamples.Server.Workflow
{
    public sealed class WfeLoanSetStateAction :
        KrWorkflowActionBase
    {
        #region Nested Types

        public class StepInfo
        {
            public int? Deadline { get; set; }
            public string TaskText { get; set; }
            public string TaskTextOnRevision { get; set; }
            public Guid ID { get; set; }
        }

        #endregion

        #region Consts

        public const string MainActionSection = "WfeLoanSetStateAction";
        public const string MainLoanSection = "WfeLACommonInfo";
        public const string StagesSection = "WfeStagesStatistics";
        public const string MainTaskSection = "WfeTasksCommonInfo";

        #endregion

        #region Fields

        private readonly IRolesFormationStrategy rolesFormationStrategy;

        #endregion

        #region Constructors

        public WfeLoanSetStateAction(
            ICardRepository cardRepository,
            IWorkflowEngineCardRequestExtender requestExtender,
            IBusinessCalendarService calendarService,
            IKrDocumentStateManager krDocumentStateManager,
            IRolesFormationStrategy rolesFormationStrategy,
            IKrHistoryStrategy historyStrategy,
            IKrWorkflowStateStrategy stateStrategy)
            : base(
                  WfeHelper.WfeLoanSetStateDescriptor,
                  cardRepository,
                  requestExtender,
                  calendarService,
                  krDocumentStateManager,
                  historyStrategy,
                  stateStrategy) =>
            this.rolesFormationStrategy = rolesFormationStrategy;

        #endregion

        #region Methods Overrides

        protected override async Task ExecuteAsync(IWorkflowEngineContext context, IWorkflowEngineCompiled scriptObject)
        {
            if (context.Signal.Type == WorkflowSignalTypes.Default)
            {
                var card = await context.GetMainCardAsync(context.CancellationToken);

                int? stepID = await context.GetAsync<int?>(MainActionSection, "Step", "ID"),
                    stageID = await context.GetAsync<int?>(MainActionSection, "Stage", "ID"),
                    stateID = await context.GetAsync<int?>(MainActionSection, "State", "ID"),
                    prevStageID = (int?) card.Sections[MainLoanSection].Fields["ProcessStagesID"];

                string stageName = await context.GetAsync<string>(MainActionSection, "Stage", "Name"),
                    stepName = await context.GetAsync<string>(MainActionSection, "Step", "Name");

                if (stageID.HasValue)
                {
                    await this.StateStrategy.SetStateIDAsync(
                        context,
                        (KrState) stateID!.Value,
                        context.ValidationResult,
                        context.CancellationToken);
                }

                if (stepID.HasValue)
                {
                    await this.CalculateAndFillTaskInfoAsync(
                        context,
                        stepID.Value);
                }

                if (!stageID.HasValue
                    || !prevStageID.HasValue
                    || stageID.Value != prevStageID.Value)
                {
                    if (prevStageID.HasValue)
                    {
                        await this.CompleteStageAsync(context, card, stageID, stageName);
                    }

                    if (stageID.HasValue)
                    {
                        await this.StartStageAsync(context, card, stageID.Value, stageName);
                    }
                }

                card.Sections[MainLoanSection].Fields["ProcessStepID"] = Int32Boxes.Box(stepID);
                card.Sections[MainLoanSection].Fields["ProcessStepName"] = stepName;
                card.Sections[MainLoanSection].Fields["ProcessStagesID"] = Int32Boxes.Box(stageID);
                card.Sections[MainLoanSection].Fields["ProcessStagesName"] = stageName;
            }
        }

        #endregion

        #region Private Method

        private async ValueTask CompleteStageAsync(
            IWorkflowEngineContext context,
            Card card,
            int? nextStageID,
            string nextStageName)
        {
            var now = DateTime.UtcNow;
            var stagesSection = card.Sections[StagesSection];
            CardRow row;
            var rowID = context.ProcessInstance.Hash.TryGet<Guid?>("StageRowID");
            if (rowID.HasValue
                && (row = stagesSection.Rows.FirstOrDefault(x => x.RowID == rowID.Value)) != null)
            {
                var startDate = row.Get<DateTime>("DateStart");

                row.Fields["DateFinish"] = now;
                row.Fields["NextStageID"] = Int32Boxes.Box(nextStageID);
                row.Fields["NextStageName"] = nextStageName;
                row.State = CardRowState.Modified;

                var taskRoleID = WorkflowEngineHelper.Get<Guid?>(context.NodeInstance.Hash, "Role", "ID");
                var completeTask = context.StoreCard.Tasks.FirstOrDefault(x => x.Action == CardTaskAction.Complete);
                if (completeTask is not null)
                {
                    var taskSection = completeTask.Card.Sections[MainTaskSection];
                    row.Fields["CauseReturnID"] = taskSection.Fields.TryGet<Guid?>("ReasonForReturnID");
                    row.Fields["CauseReturnName"] = taskSection.Fields.TryGet<string>("ReasonForReturnName");

                    var taskRole = await CardComponentHelper.TryGetMasterTaskAssignedRoleAsync(
                        completeTask,
                        context.ValidationResult,
                        typeof(KrApprovalAction),
                        context.DbScope.Db,
                        context.DbScope.BuilderFactory,
                        cancellationToken: context.CancellationToken);

                    if (taskRole is null)
                    {
                        return;
                    }

                    taskRoleID = taskRole.RoleID;
                }

                var userID = taskRoleID ?? context.Session.User.ID;
                var taskRoleCalendarInfo = await this.CalendarService.GetRoleCalendarInfoAsync(userID, context.CancellationToken);

                if (taskRoleCalendarInfo is null)
                {
                    context.ValidationResult.AddError(this, await LocalizationManager.FormatAsync("$KrMessages_NoRoleCalendar", userID));
                    return;
                }

                row.Fields["LabourIntensity"] = await this.CalendarService.GetDateDiffAsync(startDate, now, taskRoleCalendarInfo.CalendarID, cancellationToken: context.CancellationToken) * 0.25;
            }
        }

        private async Task StartStageAsync(
            IWorkflowEngineContext context,
            Card card,
            int stageID,
            string stageName)
        {
            var taskRoleID = WorkflowEngineHelper.Get<Guid>(context.NodeInstance.Hash, "Role", "ID");
            var taskRoleCalendarInfo = await this.CalendarService.GetRoleCalendarInfoAsync(taskRoleID, context.CancellationToken);

            if (taskRoleCalendarInfo is null)
            {
                context.ValidationResult.AddError(this, await LocalizationManager.FormatAsync("$KrMessages_NoRoleCalendar", taskRoleID));
                return;
            }

            var now = DateTime.UtcNow;
            var planned = await this.CalendarService.AddWorkingQuantsToDateAsync(
                now,
                await GetStageDeadlineAsync(context.DbScope, stageID, context.CancellationToken) * 4,
                taskRoleCalendarInfo.CalendarID,
                cancellationToken: context.CancellationToken);

            card.Sections[MainLoanSection].Fields["StageDataSart"] = now;
            card.Sections[MainLoanSection].Fields["StagePlaneDataFinish"] = planned;

            var stagesSection = card.Sections[StagesSection];
            var order = stagesSection.Rows.Count;

            var newRow = stagesSection.Rows.Add();
            newRow.RowID = Guid.NewGuid();
            newRow.State = CardRowState.Inserted;
            newRow.Fields["Iterator"] = Int32Boxes.Box(order);
            newRow.Fields["DateStart"] = now;
            newRow.Fields["CurrentStageID"] = Int32Boxes.Box(stageID);
            newRow.Fields["CurrentStageName"] = stageName;
            newRow.Fields["PlaneDataFinish"] = planned;
            newRow.Fields["LabourIntensity"] = DoubleBoxes.Zero;

            context.ProcessInstance.Hash["StageRowID"] = newRow.RowID;
        }

        private static async Task<int> GetStageDeadlineAsync(IDbScope dbScope, int stageID, CancellationToken cancellationToken = default)
        {
            var db = dbScope.Db;
            return await db.SetCommand(
                dbScope
                    .BuilderFactory
                    .Select()
                        .Top(1)
                        .C("Deadline")
                    .From("WfeSettingPipelineLoanStage").NoLock()
                    .Where().C("StageID").Equals().P("StageID")
                    .Limit(1)
                    .Build(),
                    db.Parameter("StageID", stageID))
                .LogCommand()
                .ExecuteAsync<int?>(cancellationToken) ?? 0;
        }

        private async Task CalculateAndFillTaskInfoAsync(IWorkflowEngineContext context, int stepID)
        {
            var db = context.DbScope.Db;

            var taskID = context.Signal.As<WorkflowEngineTaskSignal>().TaskIDs?.FirstOrDefault();
            var isRevoke = context.Signal.Hash.TryGet<bool>("IsRevoke");

            var stepInfo = await db.SetCommand(
                    context.DbScope.BuilderFactory
                        .Select()
                            .C("ID")
                            .C("TaskText")
                            .C("TaskTextOnRevision")
                            .C("Deadline")
                        .From("WfeStepConfiguration").NoLock()
                        .Where()
                            .C("StepIDID").Equals().P("StepID")
                            .And()
                            .C("SegmentID").Equals().P("SegmentID")
                        .Build(),
                    db.Parameter("StepID", stepID),
                    db.Parameter("SegmentID", context.ProcessInstance.Hash.TryGet<int>("Segment")))
                .LogCommand()
                .ExecuteAsync<StepInfo>(context.CancellationToken);

            if (stepInfo is null)
            {
                context.ValidationResult.AddError(
                    this,
                    "Не заданы настройки шага.");
                return;
            }

            context.NodeInstance.Hash["Digest"] = string.Format(
                "{0}\n\n{1}: {2}",
                isRevoke ? stepInfo.TaskTextOnRevision : stepInfo.TaskText,
                context.Session.User.Name,
                taskID.HasValue
                    ? (await context.GetTaskAsync(taskID.Value, context.CancellationToken))
                    ?.Card.Sections["WfeTasksCommonInfo"].RawFields.TryGet<string>("Comment") ?? string.Empty
                    : string.Empty);

            context.NodeInstance.Hash["Period"] = stepInfo.Deadline / 8.0;

            var roles = new Dictionary<Guid, string>();

            db.SetCommand(
                    context.DbScope.BuilderFactory
                        .Select()
                            .C("RoleID")
                            .C("RoleName")
                        .From("WfeListRoles").NoLock()
                            .Where().C("ID").Equals().P("ID")
                        .Build(),
                    db.Parameter("ID", stepInfo.ID))
                .LogCommand();

            await using (var reader = await db.ExecuteReaderAsync(context.CancellationToken))
            {
                while (await reader.ReadAsync(context.CancellationToken))
                {
                    roles[reader.GetValue<Guid>(0)] = reader.GetValue<string>(1);
                }
            }

            if (roles.Count == 0)
            {
                context.ValidationResult.AddError(
                    this,
                    "Не заданы исполнители шага.");
                return;
            }

            var (taskRoleID, taskRoleName) = await this.rolesFormationStrategy.GetRolesFormationAsync(
                context.ProcessInstance.CardID,
                roles,
                context.CancellationToken);

            WorkflowEngineHelper.Set(context.NodeInstance.Hash, taskRoleID, "Role", "ID");
            WorkflowEngineHelper.Set(context.NodeInstance.Hash, taskRoleName, "Role", "Name");
        }

        #endregion
    }
}
