﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using Tessa.Cards;
using Tessa.Cards.Extensions;
using Tessa.Platform.Data;
using Tessa.Platform.Licensing;
using Tessa.Platform.Storage;
using Tessa.Roles;
using Tessa.Roles.Acl;
using Tessa.Roles.Acl.Manager.Params;

namespace Tessa.Extensions.AclExamples.Server.Roles
{
    /// <summary>
    /// Расширение на сохранение карточки подразделения.
    /// Заполняет таблицу маппинга подразделений и типов документов.
    /// Система должна заполнять данную таблицу как для сохраняемого подразделения, так и для всех дочерних подразделений.
    /// 
    /// Производит запуск обновления ACL карточек, которые относятся к подразделениям.
    /// </summary>
    public sealed class DepartmentRoleCardStoreExtension : CardStoreExtension
    {
        #region Fields

        private const string MapSectionName = "AeDepartmentRoleTypesMap";
        private static readonly string[] sectionsWithTypes = new string[]
        {
            "AeDepartmentRoleReaders",
            "AeDepartmentRoleApprovers",
            "AeDepartmentRoleSecretaries",
        };

        private readonly IDepartmentSettingsDocTypesProvider docTypesProvider;
        private readonly IAclManager aclManager;
        private readonly ILicenseManager licenseManager;

        #endregion

        #region Constructors

        public DepartmentRoleCardStoreExtension(
            IDepartmentSettingsDocTypesProvider docTypesProvider,
            IAclManager aclManager,
            ILicenseManager licenseManager)
        {
            ThrowIfNull(docTypesProvider);
            ThrowIfNull(aclManager);
            ThrowIfNull(licenseManager);

            this.docTypesProvider = docTypesProvider;
            this.aclManager = aclManager;
            this.licenseManager = licenseManager;
        }

        #endregion

        #region Base Overrides

        public override async Task BeforeCommitTransaction(ICardStoreExtensionContext context)
        {
            if (!context.ValidationResult.IsSuccessful())
            {
                return;
            }

            await this.UpdateTypeMappingAsync(context);
        }

        #endregion

        #region Private Methods

        private async Task UpdateTypeMappingAsync(ICardStoreExtensionContext context)
        {
            if (context.Request.TryGetCard() is not { } card
                || card.Sections.Count == 0)
            {
                return;
            }

            var typesToAdd = await this.GetNewDocumentTypesAsync(card, context.CancellationToken) ?? Array.Empty<(Guid, string)>();
            if (typesToAdd.Count > 0)
            {
                var db = context.DbScope.Db;
                var builderFactory = context.DbScope.BuilderFactory;
                var getQuery =
                    builderFactory
                        .Select().C("deps", "ID")
                        .From().Function("AeGetDepartments", b => b.P("RoleID")).As("deps")
                        .LeftJoin(MapSectionName, "map").NoLock()
                            .On().C("map", "ID").Equals().C("deps", "ID")
                                .And().C("map", "TypeID").Equals().P("TypeID")
                        .Where().C("map", "ID").IsNull()
                        .Build();

                var insertQuery =
                    builderFactory
                        .InsertInto(MapSectionName, "ID", "RowID", "TypeID", "TypeCaption")
                        .Select().C("deps", "ID").NewGuid().P("TypeID").P("TypeCaption")
                        .From().Function("AeGetDepartments", b => b.P("RoleID")).As("deps")
                        .LeftJoin(MapSectionName, "map").NoLock()
                            .On().C("map", "ID").Equals().C("deps", "ID")
                                .And().C("map", "TypeID").Equals().P("TypeID")
                        .Where().C("map", "ID").IsNull()
                        .Build();

                List<Guid> allUpdateDepartmentsIDs = null;
                foreach (var (typeID, typeCaption) in typesToAdd)
                {
                    var updateDepartmentsIDs = await db.SetCommand(
                        getQuery,
                        db.Parameter("RoleID", card.ID, LinqToDB.DataType.Guid),
                        db.Parameter("TypeID", typeID, LinqToDB.DataType.Guid))
                        .LogCommand()
                        .ExecuteListAsync<Guid>(context.CancellationToken);

                    if (updateDepartmentsIDs.Count > 0)
                    {
                        allUpdateDepartmentsIDs ??= new List<Guid>();
                        allUpdateDepartmentsIDs.AddRange(updateDepartmentsIDs);
                        await db.SetCommand(
                            insertQuery,
                            db.Parameter("RoleID", card.ID, LinqToDB.DataType.Guid),
                            db.Parameter("TypeID", typeID, LinqToDB.DataType.Guid),
                            db.Parameter("TypeCaption", typeCaption, LinqToDB.DataType.Text))
                            .LogCommand()
                            .ExecuteNonQueryAsync(context.CancellationToken);
                    }
                }

                if (allUpdateDepartmentsIDs is not null)
                {
                    foreach(var depID in allUpdateDepartmentsIDs)
                    {
                        var triggerCard = new Card
                        {
                            TypeID = RoleHelper.DepartmentRoleTypeID,
                            TypeCaption = RoleHelper.DepartmentRoleTypeCaption,
                            TypeName = RoleHelper.DepartmentRoleTypeName,
                            ID = depID,
                        };
                        var newRow = triggerCard.Sections.GetOrAddTable(MapSectionName).Rows.Add();
                        newRow.RowID = Guid.NewGuid();
                        newRow.State = CardRowState.Inserted;
                        
                        // Проверка лицензии ACL.
                        ILicense license = await this.licenseManager.GetLicenseAsync(context.CancellationToken);
                        if (license.Modules.HasEnterpriseOrContains(LicenseModules.AclID))
                        {
                            var result = await this.aclManager.UpdateAclAsync(
                                new AclManagerRequest(
                                    new GetRulesByTriggerCardParam
                                    {
                                        TriggerCard = triggerCard
                                    },
                                    options: AclManagerRequestOptions.GetDefaultServerOptions()),
                                context.CancellationToken);

                            context.ValidationResult.Add(result.ValidationResult);
                            if (!result.ValidationResult.IsSuccessful)
                            {
                                return;
                            }
                        }
                    }
                }
            }
        }

        private async ValueTask<IList<(Guid, string)>> GetNewDocumentTypesAsync(
            Card card, 
            CancellationToken cancellationToken = default)
        {
            Dictionary<Guid, string> typesToAdd = null;
            foreach (var sectionName in sectionsWithTypes)
            {
                if (card.Sections.TryGetValue(sectionName, out var section))
                {
                    foreach (var row in section.Rows)
                    {
                        if (row.State == CardRowState.Inserted
                            || row.State == CardRowState.Modified)
                        {
                            var typeID = row.TryGet<Guid?>("TypeID");
                            if (typeID.HasValue)
                            {
                                typesToAdd ??= new Dictionary<Guid, string>();
                                typesToAdd[typeID.Value] = row.TryGet<string>("TypeCaption");
                            }
                            else
                            {
                                return await this.docTypesProvider.GetAllDocTypesAsync(cancellationToken);
                            }
                        }
                    }
                }
            }

            return typesToAdd?.Select(x => (x.Key, x.Value)).ToArray();
        }

        #endregion
    }
}
